--[[
    UnloadSilageAdditive

	Enables Vehicles capable of loading silage additive to unload it.

	@author: 	BayernGamers
	@date: 		20.05.2025
	@version:	1.1

	History:	v1.0 @20.05.2025 - initial implementation in FS25
				-------------------------------------------------------------------------------------------
                v1.1 @11.08.2025 - added support for Balers

	License:    Terms:
                    Usage:
                        Feel free to use this work as-is as long as you adhere to the following terms:
                    Attribution:
                        You must give appropriate credit to the original author when using this work.
                    No Derivatives:
                        You may not alter, transform, or build upon this work in any way.
                    Usage:
                        The work may be used for personal and commercial purposes, provided it is not modified or adapted.
                    Additional Clause:
                        This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "UnloadSilageAdditive.lua")

UnloadSilageAdditive = {}
UnloadSilageAdditive.MOD_DIR = g_currentModDirectory
UnloadSilageAdditive.MOD_NAME = g_currentModName

function UnloadSilageAdditive.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(FillUnit, specializations) and (SpecializationUtil.hasSpecialization(ForageWagon, specializations) or SpecializationUtil.hasSpecialization(Combine, specializations) or SpecializationUtil.hasSpecialization(Baler, specializations))
end

function UnloadSilageAdditive.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", UnloadSilageAdditive)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", UnloadSilageAdditive)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", UnloadSilageAdditive)
end

function UnloadSilageAdditive:onPreLoad()
    self.spec_unloadSilageAdditive = {}
end

function UnloadSilageAdditive:onPostLoad(savegame)
	local spec_fillUnit = self.spec_fillUnit

	if spec_fillUnit ~= nil then

        local supportsSilageAdditive = false
        for index, fillUnit in pairs(spec_fillUnit.fillUnits) do
            log:printDevInfo("Checking FillUnit " .. tostring(index) .. " for Silage Additive support", LoggingUtil.DEBUG_LEVELS.HIGH)
            if spec_fillUnit:getFillUnitSupportsFillType(index, FillType.SILAGE_ADDITIVE) then
                supportsSilageAdditive = true
                log:printDevInfo("FillUnit " .. tostring(index) .. " supports Silage Additive", LoggingUtil.DEBUG_LEVELS.HIGH)
            end
        end
        log:printDevInfo(string.format("Vehicle %s does %s support Silage Additive", self:getName(), supportsSilageAdditive and "" or "not"), LoggingUtil.DEBUG_LEVELS.HIGH)

        if supportsSilageAdditive then
            if spec_fillUnit.unloading == nil then
                spec_fillUnit.unloading = {}
            end

            if #spec_fillUnit.unloading == 0 then
                local unloadNodeBack = createTransformGroup("unloadNodeBack")
                local unloadNodeLeft = createTransformGroup("unloadNodeLeft")
                local unloadNodeRight = createTransformGroup("unloadNodeRight")
                
                for index, component in ipairs(self.components) do
                    log:printDevInfo("Found Component: " .. tostring(index) .. " with node: " .. tostring(component.node), LoggingUtil.DEBUG_LEVELS.HIGH)
                    if component.node ~= nil then
                        link(component.node, unloadNodeBack)
                        link(component.node, unloadNodeLeft)
                        link(component.node, unloadNodeRight)
                        break
                    end
                end

                local lengthOffset = self.xmlFile:getValue("vehicle.base.size#length")
                local widthOffset = self.xmlFile:getValue("vehicle.base.size#width")
                if lengthOffset == nil then
                    lengthOffset = 5
                else
                    lengthOffset = lengthOffset * 0.5 + 0.75
                end

                if widthOffset == nil then
                    widthOffset = 5
                else
                    widthOffset = widthOffset * 0.5 + 1.5
                end
                
                log:printDevInfo("LengthOffset: -" .. tostring(lengthOffset), LoggingUtil.DEBUG_LEVELS.HIGH)
                log:printDevInfo("WidthOffset: " .. tostring(widthOffset), LoggingUtil.DEBUG_LEVELS.HIGH)

                setTranslation(unloadNodeBack, 0, 0.2, -lengthOffset)
                setTranslation(unloadNodeLeft, widthOffset, 0.2, 0)
                setTranslation(unloadNodeRight, -widthOffset, 0.2, 0)

                setRotation(unloadNodeLeft, 0, math.rad(90), 0)
                setRotation(unloadNodeRight, 0, math.rad(90), 0)

                local unloadingBack =  {
                    node = unloadNodeBack,
                    offset = {0, 0, 0},
                    width = 6
                }

                local unloadingLeft =  {
                    node = unloadNodeLeft,
                    offset = {0, 0, 0},
                    width = 10
                }

                local unloadingRight =  {
                    node = unloadNodeRight,
                    offset = {0, 0, 0},
                    width = 10
                }

                table.insert(spec_fillUnit.unloading, unloadingRight)
                table.insert(spec_fillUnit.unloading, unloadingBack)
                table.insert(spec_fillUnit.unloading, unloadingLeft)

                log:printDevInfo("Created Unloading Node Back " .. tostring(unloadingBack), LoggingUtil.DEBUG_LEVELS.HIGH)
                log:printDevInfo("Values: " .. tostring(unloadingBack.node) .. " " .. tostring(unloadingBack.offset) .. " " .. tostring(unloadingBack.width), LoggingUtil.DEBUG_LEVELS.HIGH)

                log:printDevInfo("Created Unloading Node Left " .. tostring(unloadingLeft), LoggingUtil.DEBUG_LEVELS.HIGH)
                log:printDevInfo("Values: " .. tostring(unloadingLeft.node) .. " " .. tostring(unloadingLeft.offset) .. " " .. tostring(unloadingLeft.width), LoggingUtil.DEBUG_LEVELS.HIGH)

                log:printDevInfo("Created Unloading Node Right " .. tostring(unloadingRight), LoggingUtil.DEBUG_LEVELS.HIGH)
                log:printDevInfo("Values: " .. tostring(unloadingRight.node) .. " " .. tostring(unloadingRight.offset) .. " " .. tostring(unloadingRight.width), LoggingUtil.DEBUG_LEVELS.HIGH)
            end
        end
	end
end

function UnloadSilageAdditive:onUpdate(dt)
    local spec_fillUnit = self.spec_fillUnit

    if spec_fillUnit ~= nil then
        for index, fillUnit in pairs(spec_fillUnit.fillUnits) do
            log:printDevInfo("Checking FillUnit " .. tostring(index) .. " for unloading", LoggingUtil.DEBUG_LEVELS.LOW)
            log:printDevInfo("FillUnit can be unloaded: " .. tostring(fillUnit.canBeUnloaded), LoggingUtil.DEBUG_LEVELS.LOW)

            if spec_fillUnit:getFillUnitSupportsFillType(index, FillType.SILAGE_ADDITIVE) then
                if spec_fillUnit:getFillUnitFillLevel(index) > 0 then
                    fillUnit.canBeUnloaded = true
                    log:printDevInfo("FillUnit " .. tostring(index) .. " can be unloaded", LoggingUtil.DEBUG_LEVELS.LOW)
                else
                    fillUnit.canBeUnloaded = false
                    log:printDevInfo("FillUnit " .. tostring(index) .. " cannot be unloaded", LoggingUtil.DEBUG_LEVELS.LOW)
                end
            end
        end
    end
end