-- ============================================================================
-- === SpeedMeterDisplayExt.Lua
-- === Mod by [LSMT] Modding Team 
-- === LS25 /FS25
-- === Script by [LSMT] BaTt3RiE @ 2025
-- === Ver 1.1.0.0
-- ============================================================================

SpeedMeterDisplayExt = {}

-- Debug-Optionen
local DEBUG = false         -- Setze auf true, um Debug-Ausgaben zu aktivieren
local function dprint(...)
    if DEBUG then
        print("[SpeedMeterDisplayExt] " .. string.format(...))
    end
end

-- Mod‑Verzeichnis und Grafikpfade
local ModDirectory = g_currentModDirectory or ""
local hudAtlasXmlPath = ModDirectory .. "hud/adblue_hud.xml"
local hudAtlasPrefix = "adblue"

dprint("Mod-Verzeichnis: %s", ModDirectory)
dprint("HUD-Atlas XML-Pfad: %s", hudAtlasXmlPath)


-- Konfiguration AdBlue Anzeige Gauge
SpeedMeterDisplayExt.DEFAULT_POSITION = {x = -450, y = 0}
SpeedMeterDisplayExt.DEFAULT_SIZE = {width = 64, height = 128}          -- Größe in Pixeln (Orginal)

-- Hintergrund Konfiguration
SpeedMeterDisplayExt.BACKGROUND_IMAGE_SIZE = {width = 80, height = 200} -- Größe des Hintergrunds
SpeedMeterDisplayExt.BACKGROUND_IMAGE_OFFSET = {x = -5, y = -25}

-- Zweite Gauge-Konfiguration
SpeedMeterDisplayExt.GAUGE2_SIZE = {width = 30, height = 140}           -- Größe der zweiten Anzeige
SpeedMeterDisplayExt.GAUGE2_OFFSET = {x = 20, y = -10}                  -- Position relativ zum Hauptbild

-- Tachonadel Konfiguration
SpeedMeterDisplayExt.SECOND_IMAGE_SIZE = {width = 128, height = 128}    -- Größe der Nadel
SpeedMeterDisplayExt.SECOND_IMAGE_OFFSET = {x = 0, y = 0}

-- Progressbar Konfiguration Background
SpeedMeterDisplayExt.PROGRESSBAR_SIZE = {width = 8, height = 140}       -- Größe aus deinem Atlas
SpeedMeterDisplayExt.PROGRESSBAR_OFFSET = {x = 30, y = -10}             -- Offset anpassen nach Bedarf

-- Fill-Progressbar Konfiguration
SpeedMeterDisplayExt.PROGRESSBAR2_SIZE = {width = 8, height = 140}      -- Größe der zweiten Progressbar
SpeedMeterDisplayExt.PROGRESSBAR2_OFFSET = {x = 30, y = -10}            -- Offset für die zweite Progressbar

-- Prozentanzeige Konfiguration
SpeedMeterDisplayExt.PERCENT_TEXT_OFFSET = {x = 40, y = 50}             -- Position der Prozentanzeige
SpeedMeterDisplayExt.PERCENT_TEXT_SIZE = 0.025                          -- Textgröße

-- Prozentanzeige Adblue Icon
SpeedMeterDisplayExt.ADDITIONAL_IMAGE_SIZE = {width = 32, height = 32}  -- Größe des zusätzlichen Bildes
SpeedMeterDisplayExt.ADDITIONAL_IMAGE_OFFSET = {x = 20, y = 80}         -- Position relativ zum Hauptbild

-- AdBlue Fuel-Icon Konfiguration
SpeedMeterDisplayExt.FUEL_ICON_SIZE = {width = 20, height = 20}         -- Größe des Fuel Icons
SpeedMeterDisplayExt.FUEL_ICON_OFFSET = {x = 25, y = 150}               -- Position des Icons in Bezug auf das Hauptbild

-- AdBlue Control Light Konfiguration
SpeedMeterDisplayExt.FUEL_ICON2_SIZE = {width = 10, height = 10}        -- Größe des Fuel Icons
SpeedMeterDisplayExt.FUEL_ICON2_OFFSET = {x = 10, y = 130}              -- Position des Icons in Bezug auf das Hauptbild

-- Dreiecke Konfiguration
SpeedMeterDisplayExt.TRIANGLE_LEFT_SIZE = {width = 12, height = 12}     -- Größe des linken Dreiecks
SpeedMeterDisplayExt.TRIANGLE_LEFT_OFFSET = {x = 10, y = 150}           -- Position relativ zum Hauptbild

SpeedMeterDisplayExt.TRIANGLE_RIGHT_SIZE = {width = 12, height = 12}    -- Größe des rechten Dreiecks
SpeedMeterDisplayExt.TRIANGLE_RIGHT_OFFSET = {x = 50, y = 150}          -- Position relativ zum Hauptbild

-- ProgressBarSquare-Konfiguration
SpeedMeterDisplayExt.SQUARES_COUNT = 10                                 -- Anzahl der Quadrate
SpeedMeterDisplayExt.SQUARE_SIZE = {width = 16, height = 8}             -- Größe eines Quadrats
SpeedMeterDisplayExt.SQUARES_START_OFFSET = {x = 25, y = 5}             -- Position des ersten Quadrats
SpeedMeterDisplayExt.SQUARES_GAP = 4                                    -- Abstand zwischen den Quadraten
SpeedMeterDisplayExt.SHOW_SQUARES = false                               -- Quadrate standardmäßig anzeigen?

-- Zweite ProgressBarSquare-Konfiguration
SpeedMeterDisplayExt.SQUARES2_COUNT = 12                                -- Anzahl der zweiten Quadrate
SpeedMeterDisplayExt.SQUARE2_SIZE = {width = 16, height = 8}            -- Größe eines Quadrats
SpeedMeterDisplayExt.SQUARES2_START_OFFSET = {x = 30, y = -15}          -- Position des ersten Quadrats
SpeedMeterDisplayExt.SQUARES2_GAP = 4                                   -- Abstand zwischen den Quadraten
SpeedMeterDisplayExt.SHOW_SQUARES2 = false                              -- Zweite Quadratreihe standardmäßig anzeigen?

SpeedMeterDisplayExt.BACKGROUND2_IMAGE_SIZE = {width = 70, height = 155}-- Größe des Hintergrunds
SpeedMeterDisplayExt.BACKGROUND2_IMAGE_OFFSET = {x = 0, y = -10}
-- Anzeige-Einstellungen

-- Immer anzeigen?
SpeedMeterDisplayExt.SHOW_BACKGROUND = true         -- Hintergrund anzeigen?
SpeedMeterDisplayExt.SHOW_BACKGROUND2 = true        -- Hintergrund anzeigen?
SpeedMeterDisplayExt.SHOW_TRIANGLE_LEFT = true      -- Linkes Dreieck anzeigen?
SpeedMeterDisplayExt.SHOW_TRIANGLE_RIGHT = true     -- Rechtes Dreieck anzeigen?

SpeedMeterDisplayExt.SHOW_GAUGE = false             -- AdBlue-Anzeige anzeigen?
SpeedMeterDisplayExt.SHOW_NEEDLE = false            -- Nadel anzeigen?

SpeedMeterDisplayExt.SHOW_PROGRESSBAR = true        -- Progressbar anzeigen?
SpeedMeterDisplayExt.SHOW_PROGRESSBAR2 = true       -- Zweite Progressbar anzeigen?
SpeedMeterDisplayExt.SHOW_FUEL_ICON = true          -- AdBlue Icon anzeigen?

SpeedMeterDisplayExt.SHOW_PERCENT_TEXT = false      -- Prozentanzeige für AdBlue anzeigen?
SpeedMeterDisplayExt.SHOW_ADDITIONAL_IMAGE = false  -- Adblue Icon anzeigen?

-- Skalierungskonfiguration für unterschiedliche Auflösungen 
SpeedMeterDisplayExt.UI_SCALE = 1.0                 -- Basis-Skalierungsfaktor (1.0 = keine zusätzliche Skalierung)

SpeedMeterDisplayExt.MOVE_STEP = 5                  -- Pixel pro Tastendruck
SpeedMeterDisplayExt.isPositioningMode = false      -- Flag für Positionierungsmodus
SpeedMeterDisplayExt.CONFIG_FILENAME = getUserProfileAppPath() .. "modSettings/AdBlueHUDSettings.xml" -- Datei zum Speichern im modSettings-Verzeichnis

-- Flag zur Kontrolle, ob Icons schon initialisiert wurden
local iconsInitialized = false
local hudAtlasLoaded = false

-- Zusätzliche Variablen für die Maussteuerung im Positionierungsmodus
SpeedMeterDisplayExt.isDragging = false  -- Wird verwendet um zu verfolgen, ob aktuell gezogen wird
SpeedMeterDisplayExt.dragStartX = 0      -- Mausposition bei Start des Ziehens
SpeedMeterDisplayExt.dragStartY = 0      -- Mausposition bei Start des Ziehens
SpeedMeterDisplayExt.dragOffsetX = 0     -- Offset zwischen Mausposition und Anzeigeposition
SpeedMeterDisplayExt.dragOffsetY = 0     -- Offset zwischen Mausposition und Anzeigeposition

-- Original-Funktionen speichern
SpeedMeterDisplayExt.originalNew = SpeedMeterDisplay.new
SpeedMeterDisplayExt.originalDelete = SpeedMeterDisplay.delete
SpeedMeterDisplayExt.originalStoreScaledValues = SpeedMeterDisplay.storeScaledValues
SpeedMeterDisplayExt.originalDraw = SpeedMeterDisplay.draw
SpeedMeterDisplayExt.originalUpdate = SpeedMeterDisplay.update

-- Design-Definitionen
SpeedMeterDisplayExt.DESIGNS = {
    {
        name = "Klassisch",
        settings = {
            SHOW_GAUGE = true,
            SHOW_NEEDLE = true,
            SHOW_PROGRESSBAR = false,
            SHOW_PROGRESSBAR2 = false,
            SHOW_FUEL_ICON = false,
            SHOW_FUEL_ICON2 = true,
            SHOW_BACKGROUND = true,
            SHOW_BACKGROUND2 = false,
            SHOW_TRIANGLE_LEFT = true,
            SHOW_TRIANGLE_RIGHT = true,
            SHOW_SQUARES = false,
            SHOW_SQUARES2 = false,
            SHOW_PERCENT_TEXT = false,
            SHOW_ADDITIONAL_IMAGE = false,
        }
    },
    {
        name = "Modern",
        settings = {
            SHOW_GAUGE = false,
            SHOW_NEEDLE = false,
            SHOW_PROGRESSBAR = true,
            SHOW_PROGRESSBAR2 = true,
            SHOW_FUEL_ICON = true,
            SHOW_FUEL_ICON2 = true,
            SHOW_BACKGROUND = true,
            SHOW_BACKGROUND2 = false,
            SHOW_TRIANGLE_LEFT = true,
            SHOW_TRIANGLE_RIGHT = true,
            SHOW_SQUARES = false,
            SHOW_SQUARES2 = false,
            SHOW_PERCENT_TEXT = false,
            SHOW_ADDITIONAL_IMAGE = false,
        }
    },
    {
        name = "Modern 2",
        settings = {
            SHOW_GAUGE = false,
            SHOW_NEEDLE = false,
            SHOW_PROGRESSBAR = false,
            SHOW_PROGRESSBAR2 = false,
            SHOW_FUEL_ICON = false,
            SHOW_FUEL_ICON2 = false,
            SHOW_BACKGROUND = true,
            SHOW_BACKGROUND2 = true,
            SHOW_TRIANGLE_LEFT = true,
            SHOW_TRIANGLE_RIGHT = true,
            SHOW_SQUARES = false,
            SHOW_SQUARES2 = true,
            SHOW_PERCENT_TEXT = false,
            SHOW_ADDITIONAL_IMAGE = false,
        }
    },
    {
        name = "Modern Prozent Anzeige",
        settings = {
            SHOW_GAUGE = false,
            SHOW_NEEDLE = false,
            SHOW_PROGRESSBAR = false,
            SHOW_PROGRESSBAR2 = false,
            SHOW_FUEL_ICON = false,
            SHOW_FUEL_ICON2 = false,
            SHOW_BACKGROUND = true,
            SHOW_BACKGROUND2 = false,
            SHOW_TRIANGLE_LEFT = true,
            SHOW_TRIANGLE_RIGHT = true,
            SHOW_SQUARES = false,
            SHOW_SQUARES2 = false,
            SHOW_PERCENT_TEXT = true,
            SHOW_ADDITIONAL_IMAGE = true
        }
    }
}

-- Aktives Design
SpeedMeterDisplayExt.currentDesignIndex = 2 -- Standardmäßig auf "Modern" setzen
SpeedMeterDisplayExt.squareImages = {} -- Array für die Quadrat-Overlays
SpeedMeterDisplayExt.square2Images = {} -- Array für die zweite Reihe von Quadrat-Overlays

-- Atlas laden
function SpeedMeterDisplayExt.loadHudAtlas()
    if hudAtlasLoaded then return true end
    
    dprint("Lade HUD-Atlas aus %s", hudAtlasXmlPath)
    if g_gui and g_overlayManager then
        g_overlayManager:addTextureConfigFile(hudAtlasXmlPath, hudAtlasPrefix)
        hudAtlasLoaded = true
        return true
    else
        dprint("FEHLER: Konnte Atlas nicht laden - g_overlayManager nicht verfügbar")
        return false
    end
end

function SpeedMeterDisplayExt.getVehicleAdBlueLevelAndCapacity(vehicle)
    if vehicle.spec_adblueConsumer and vehicle.spec_adblueConsumer.hasAdBlue then
        local spec = vehicle.spec_adblueConsumer
        if spec.fillUnitIndex then
            local lvl = vehicle:getFillUnitFillLevel(spec.fillUnitIndex)
            local cap = vehicle:getFillUnitCapacity(spec.fillUnitIndex)
            return lvl, cap
        end
    end
    return nil, nil
end

-- Einmaliges Anlegen/Cachen von Custom‑Icons
function SpeedMeterDisplayExt.initCustomIcons(self)
    if iconsInitialized then return end
    
    dprint("Initialisiere benutzerdefinierte Icons aus Atlas...")
    
    -- Stelle sicher, dass der Atlas geladen ist
    if not SpeedMeterDisplayExt.loadHudAtlas() then
        dprint("FEHLER: Konnte HUD-Atlas nicht laden")
        return
    end
    
    -- Hintergrund-Overlay aus dem Atlas erstellen
    self.backgroundImage = nil
    self.backgroundImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_background", 0, 0, 0, 0)
    
    -- Prüfen, ob das Hintergrund-Overlay erfolgreich erstellt wurde
    if self.backgroundImage and self.backgroundImage.overlayId and self.backgroundImage.overlayId ~= 0 then
        dprint("Hintergrund-Bild erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des Hintergrund-Bilds aus Atlas.")
    end
    
    -- AdBlue-Anzeige aus dem Atlas erstellen
    self.customImage = nil
    self.customImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_gauge", 0, 0, 0, 0)
    
    -- Prüfen, ob das Overlay erfolgreich erstellt wurde
    if self.customImage and self.customImage.overlayId and self.customImage.overlayId ~= 0 then
        dprint("AdBlue-Anzeige erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden der AdBlue-Anzeige aus Atlas.")
    end
    
    -- Nadel-Overlay aus dem Atlas erstellen
    self.secondImage = nil
    self.secondImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_needle", 0, 0, 0, 0)
    
    -- Prüfen, ob das zweite Overlay erfolgreich erstellt wurde
    if self.secondImage and self.secondImage.overlayId and self.secondImage.overlayId ~= 0 then
        dprint("Nadel-Bild erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des Nadel-Bilds aus Atlas.")
    end
    
    self.progressbarMainImage = nil
    self.progressbarMainImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_progressbar_main", 0, 0, 0, 0)

    -- Prüfen, ob das Progressbar-Overlay erfolgreich erstellt wurde
    if self.progressbarMainImage and self.progressbarMainImage.overlayId and self.progressbarMainImage.overlayId ~= 0 then
        dprint("Progressbar-Bild erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des Progressbar-Bilds aus Atlas.")
    end

    -- Zweite Progressbar aus dem Atlas erstellen
    self.progressbarSecondImage = nil
    self.progressbarSecondImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_progressbar_main_fill", 0, 0, 0, 0)

    -- Prüfen, ob das zweite Progressbar-Overlay erfolgreich erstellt wurde
    if self.progressbarSecondImage and self.progressbarSecondImage.overlayId and self.progressbarSecondImage.overlayId ~= 0 then
        dprint("Zweite Progressbar erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden der zweiten Progressbar aus Atlas.")
    end

    -- AdBlue Icon aus dem Atlas erstellen
    self.fuelIconImage = nil
    self.fuelIconImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_fuel_icon", 0, 0, 0, 0)

    -- Prüfen, ob das Icon erfolgreich erstellt wurde
    if self.fuelIconImage and self.fuelIconImage.overlayId and self.fuelIconImage.overlayId ~= 0 then
        dprint("AdBlue-Icon erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des AdBlue-Icons aus Atlas.")
    end

    -- Linkes Dreieck aus dem Atlas erstellen
    self.triangleLeftImage = nil
    self.triangleLeftImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_triangel_left", 0, 0, 0, 0)

    -- Prüfen, ob das linke Dreieck erfolgreich erstellt wurde
    if self.triangleLeftImage and self.triangleLeftImage.overlayId and self.triangleLeftImage.overlayId ~= 0 then
        dprint("Adblue-Dreieck links erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des linken Adblue-Dreiecks aus Atlas.")
    end

    -- Rechtes Dreieck aus dem Atlas erstellen
    self.triangleRightImage = nil
    self.triangleRightImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_triangel_right", 0, 0, 0, 0)

    -- Prüfen, ob das rechte Dreieck erfolgreich erstellt wurde
    if self.triangleRightImage and self.triangleRightImage.overlayId and self.triangleRightImage.overlayId ~= 0 then
        dprint("Adblue-Dreieck rechts erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des rechten Adblue-Dreiecks aus Atlas.")
    end

    -- Squares für die Fortschrittsanzeige erstellen
    self.squareImages = {}
    for i = 1, SpeedMeterDisplayExt.SQUARES_COUNT do
        local squareImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_ProgressBarSquare", 0, 0, 0, 0)
        
        -- Prüfen, ob das Overlay erfolgreich erstellt wurde
        if squareImage and squareImage.overlayId and squareImage.overlayId ~= 0 then
            dprint("Square " .. i .. " erfolgreich aus Atlas geladen")
            table.insert(self.squareImages, squareImage)
        else
            dprint("Fehler beim Laden von Square " .. i .. " aus Atlas.")
        end
    end

    self.square2Images = {}
    for i = 1, SpeedMeterDisplayExt.SQUARES2_COUNT do
        local square2Image = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_ProgressBarSquare_Fill", 0, 0, 0, 0)
        
        -- Prüfen, ob das Overlay erfolgreich erstellt wurde
        if square2Image and square2Image.overlayId and square2Image.overlayId ~= 0 then
            dprint("Square2 " .. i .. " erfolgreich aus Atlas geladen")
            table.insert(self.square2Images, square2Image)
        else
            dprint("Fehler beim Laden von Square2 " .. i .. " aus Atlas.")
        end
    end

    -- AdBlue Icon 2 aus dem Atlas erstellen
    self.fuelIcon2Image = nil
    self.fuelIcon2Image = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_controllight", 0, 0, 0, 0)

    -- Prüfen, ob das Icon erfolgreich erstellt wurde
    if self.fuelIcon2Image and self.fuelIcon2Image.overlayId and self.fuelIcon2Image.overlayId ~= 0 then
        dprint("AdBlue-Icon 2 erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des AdBlue-Icons 2 aus Atlas.")
    end

    -- Hintergrund-Overlay aus dem Atlas erstellen
    self.background2Image = nil
    self.background2Image = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_background2", 0, 0, 0, 0)
    
    -- Prüfen, ob das Hintergrund-Overlay erfolgreich erstellt wurde
    if self.background2Image and self.background2Image.overlayId and self.background2Image.overlayId ~= 0 then
        dprint("Hintergrund-Bild 2 erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des Hintergrund-Bilds 2 aus Atlas.")
    end

    -- Zusätzliches Adblue Bild aus dem Atlas erstellen
    self.additionalImage = nil
    self.additionalImage = g_overlayManager:createOverlay(hudAtlasPrefix..".adblue_fuel_icon", 0, 0, 0, 0)

    -- Prüfen, ob das Overlay erfolgreich erstellt wurde
    if self.additionalImage and self.additionalImage.overlayId and self.additionalImage.overlayId ~= 0 then
        dprint("Zusätzliches Bild erfolgreich aus Atlas geladen")
    else
        dprint("Fehler beim Laden des zusätzlichen Bildes aus Atlas.")
    end

    -- Lade gespeicherte Position, wenn vorhanden
    SpeedMeterDisplayExt.loadSavedPosition()
    
    iconsInitialized = true
end

-- SpeedMeterDisplay.new erweitern
function SpeedMeterDisplay.new()
    dprint("Erstelle erweiterten SpeedMeterDisplay...")
    local self = SpeedMeterDisplayExt.originalNew()
    
    -- Initialisiere Icons direkt
    SpeedMeterDisplayExt.initCustomIcons(self)
    
    -- Tastenstatus initialisieren
    self.keyYWasPressed = false
    self.keyUpWasPressed = false
    self.keyDownWasPressed = false
    self.keyLeftWasPressed = false
    self.keyRightWasPressed = false
    self.keySaveWasPressed = false
    
    self.leftTriangleWasClicked = false
    self.rightTriangleWasClicked = false
    return self
end

-- SpeedMeterDisplay.delete erweitern
function SpeedMeterDisplay:delete()
    dprint("Lösche erweiterten SpeedMeterDisplay...")
    
    -- Hintergrundbild löschen
    if self.backgroundImage then
        self.backgroundImage:delete()
        self.backgroundImage = nil
    end
    
    -- Unser erstes Bild löschen
    if self.customImage then
        self.customImage:delete()
        self.customImage = nil
    end
    
    -- HINZUGEFÜGT: Zweites Bild löschen
    if self.secondImage then
        self.secondImage:delete()
        self.secondImage = nil
    end

    -- Progressbar löschen
    if self.progressbarMainImage then
        self.progressbarMainImage:delete()
        self.progressbarMainImage = nil
    end

    -- Zweite Progressbar löschen
    if self.progressbarSecondImage then
        self.progressbarSecondImage:delete()
        self.progressbarSecondImage = nil
    end

    -- AdBlue Fuel-Icon löschen
    if self.fuelIconImage then
        self.fuelIconImage:delete()
        self.fuelIconImage = nil
    end

    -- Linkes Dreieck löschen
    if self.triangleLeftImage then
        self.triangleLeftImage:delete()
        self.triangleLeftImage = nil
    end

    -- Rechtes Dreieck löschen
    if self.triangleRightImage then
        self.triangleRightImage:delete()
        self.triangleRightImage = nil
    end

    -- Squares löschen
    if self.squareImages then
        for i, square in ipairs(self.squareImages) do
            square:delete()
        end
        self.squareImages = nil
    end

    -- Zweite Reihe Squares löschen
    if self.square2Images then
        for i, square in ipairs(self.square2Images) do
            square:delete()
        end
        self.square2Images = nil
    end

    -- AdBlue Fuel-Icon 2 löschen
    if self.fuelIcon2Image then
        self.fuelIcon2Image:delete()
        self.fuelIcon2Image = nil
    end

    -- Hintergrundbild 2 löschen
    if self.background2Image then
        self.background2Image:delete()
        self.background2Image = nil
    end

    -- Zusätzliches Adblue Bild löschen
    if self.additionalImage then
        self.additionalImage:delete()
        self.additionalImage = nil
    end
    -- Original aufrufen
    SpeedMeterDisplayExt.originalDelete(self)
end

-- SpeedMeterDisplay.storeScaledValues erweitern
function SpeedMeterDisplay:storeScaledValues()
    -- Original aufrufen
    SpeedMeterDisplayExt.originalStoreScaledValues(self)
    
    -- Icons initialisieren (falls noch nicht geschehen)
    if not iconsInitialized then
        SpeedMeterDisplayExt.initCustomIcons(self)
    end
    
    -- Unsere eigene Bildskalierung mit den Spiel-Funktionen
    if self.customImage then
        dprint("Skaliere HUD-Bilder mit Spielfunktionen...")
        
        -- Größe berechnen und anwenden mit Spiel-Skalierungsfunktionen
        local width = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.DEFAULT_SIZE.width)
        local height = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.DEFAULT_SIZE.height)
        self.customImage:setDimension(width, height)
        
        -- Position relativ zum Tacho
        self.customImageOffsetX = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.DEFAULT_POSITION.x)
        self.customImageOffsetY = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.DEFAULT_POSITION.y)
        
        dprint("Bild skaliert: %dx%d @ (%d,%d)", 
               width, height, self.customImageOffsetX, self.customImageOffsetY)
        
        -- Hintergrundbild skalieren
        if self.backgroundImage then
            local bgWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.BACKGROUND_IMAGE_SIZE.width)
            local bgHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.BACKGROUND_IMAGE_SIZE.height)
            self.backgroundImage:setDimension(bgWidth, bgHeight)
            
            dprint("Hintergrundbild skaliert: %dx%d", bgWidth, bgHeight)
        end

        -- Nadel skalieren
        if self.secondImage then
            local secondWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SECOND_IMAGE_SIZE.width)
            local secondHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SECOND_IMAGE_SIZE.height)
            self.secondImage:setDimension(secondWidth, secondHeight)
            
            dprint("Nadel skaliert: %dx%d", secondWidth, secondHeight)
        end

        -- Progressbar skalieren
        if self.progressbarMainImage then
            local progressWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.PROGRESSBAR_SIZE.width)
            local progressHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.PROGRESSBAR_SIZE.height)
            self.progressbarMainImage:setDimension(progressWidth, progressHeight)
            
            dprint("Progressbar skaliert: %dx%d", progressWidth, progressHeight)
        end

        -- Zweite Progressbar Fill skalieren
        if self.progressbarSecondImage then
            local progress2Width = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.PROGRESSBAR2_SIZE.width)
            local progress2Height = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.PROGRESSBAR2_SIZE.height)
            self.progressbarSecondImage:setDimension(progress2Width, progress2Height)
            
            dprint("Zweite Progressbar skaliert: %dx%d", progress2Width, progress2Height)
        end

        -- AdBlue Fuel-Icon skalieren
        if self.fuelIconImage then
            local iconWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.FUEL_ICON_SIZE.width)
            local iconHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.FUEL_ICON_SIZE.height)
            self.fuelIconImage:setDimension(iconWidth, iconHeight)
            
            dprint("AdBlue-Icon skaliert: %dx%d", iconWidth, iconHeight)
        end

        -- Linkes Dreieck skalieren
        if self.triangleLeftImage then
            local triangleLeftWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_LEFT_SIZE.width)
            local triangleLeftHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_LEFT_SIZE.height)
            self.triangleLeftImage:setDimension(triangleLeftWidth, triangleLeftHeight)
            
            dprint("Dreieck links skaliert: %dx%d", triangleLeftWidth, triangleLeftHeight)
        end

        -- Rechtes Dreieck skalieren
        if self.triangleRightImage then
            local triangleRightWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_RIGHT_SIZE.width)
            local triangleRightHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_RIGHT_SIZE.height)
            self.triangleRightImage:setDimension(triangleRightWidth, triangleRightHeight)
            
            dprint("Dreieck rechts skaliert: %dx%d", triangleRightWidth, triangleRightHeight)
        end

        -- Squares skalieren
        if self.squareImages then
            for i, square in ipairs(self.squareImages) do
                local squareWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SQUARE_SIZE.width)
                local squareHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARE_SIZE.height)
                square:setDimension(squareWidth, squareHeight)
            end
            dprint("Squares skaliert: %dx%d", 
                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SQUARE_SIZE.width),
                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARE_SIZE.height))
        end

        if self.square2Images then
            for i, square in ipairs(self.square2Images) do
                local square2Width = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SQUARE2_SIZE.width)
                local square2Height = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARE2_SIZE.height)
                square:setDimension(square2Width, square2Height)
            end
            dprint("Squares2 skaliert: %dx%d", 
                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SQUARE2_SIZE.width),
                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARE2_SIZE.height))
        end

        -- AdBlue Fuel-Icon 2 skalieren
        if self.fuelIcon2Image then
            local icon2Width = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.FUEL_ICON2_SIZE.width)
            local icon2Height = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.FUEL_ICON2_SIZE.height)
            self.fuelIcon2Image:setDimension(icon2Width, icon2Height)
            
            dprint("AdBlue-Icon 2 skaliert: %dx%d", icon2Width, icon2Height)
        end

        -- Backgrund 2 skalieren
        if self.background2Image then
            local background2Width = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.BACKGROUND2_IMAGE_SIZE.width)
            local background2Height = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.BACKGROUND2_IMAGE_SIZE.height)
            self.background2Image:setDimension(background2Width, background2Height)
            
            dprint("Backgrund 2 skaliert: %dx%d", background2Width, background2Height)
        end

        -- Zusätzliches Adblue Bild skalieren
        if self.additionalImage then
            local additionalWidth = self:scalePixelToScreenWidth(SpeedMeterDisplayExt.ADDITIONAL_IMAGE_SIZE.width)
            local additionalHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.ADDITIONAL_IMAGE_SIZE.height)
            self.additionalImage:setDimension(additionalWidth, additionalHeight)
            
            dprint("Zusätzliches Bild skaliert: %dx%d", additionalWidth, additionalHeight)
        end

    end
end

function SpeedMeterDisplay:update(dt)
    if SpeedMeterDisplayExt.originalUpdate ~= nil then
        SpeedMeterDisplayExt.originalUpdate(self, dt)
    end
    
    if Input.isKeyPressed(Input.KEY_y) and Input.isKeyPressed(Input.KEY_lctrl) and not self.keyYWasPressed then
        self.keyYWasPressed = true
        
        -- Prüfen, ob die AdBlue-Anzeige aktuell sichtbar ist
        local shouldShowHUD = false
        
        -- Wenn bereits im Positionierungsmodus, erlauben wir immer das Deaktivieren
        if SpeedMeterDisplayExt.isPositioningMode then
            shouldShowHUD = true
        else
            -- Prüfen, ob das aktuelle Fahrzeug AdBlue hat
            if self.vehicle ~= nil and self.isVehicleDrawSafe then
                local adLvl, adCap = SpeedMeterDisplayExt.getVehicleAdBlueLevelAndCapacity(self.vehicle)
                shouldShowHUD = adLvl ~= nil and adCap ~= nil and adCap > 0
            end
        end
        
        -- Positionierungsmodus nur umschalten, wenn die HUD-Anzeige sichtbar sein sollte
        if shouldShowHUD then
            SpeedMeterDisplayExt.isPositioningMode = not SpeedMeterDisplayExt.isPositioningMode
            g_inputBinding:setShowMouseCursor(SpeedMeterDisplayExt.isPositioningMode)
            print(("HUD-Positionierungsmodus: %s"):format(tostring(SpeedMeterDisplayExt.isPositioningMode)))
        else
            print("HUD-Positionierungsmodus kann nur aktiviert werden, wenn die AdBlue-Anzeige sichtbar ist!")
        end
    elseif not Input.isKeyPressed(Input.KEY_y) and not Input.isKeyPressed(Input.KEY_lctrl) and self.keyYWasPressed then
        self.keyYWasPressed = false
    end

    do
        local veh = g_currentMission and g_currentMission.controlledVehicle
        if veh ~= nil then
            g_inputBinding:setActionEventsActiveByTarget(
                veh,
                not SpeedMeterDisplayExt.isPositioningMode
            )
        end
    end

    local isMouseVisible = g_inputBinding:getShowMouseCursor()
    local isInMenu = g_gui:getIsGuiVisible()
    if (SpeedMeterDisplayExt.isPositioningMode or (isMouseVisible and not isInMenu)) and self.customImage then
        local mouseX, mouseY = g_inputBinding:getMousePosition()   -- Bildschirmkoordinaten (0-1)
        local posX,  posY    = self:getPosition()                  -- HUD-Anker (safe-frame)

        local hudX = posX + self.customImageOffsetX
        local hudY = posY + self.customImageOffsetY

        local isMouseOverHUD =
            mouseX >= hudX and mouseX <= hudX + self.customImage.width and
            mouseY >= hudY and mouseY <= hudY + self.customImage.height

        local isPressed = Input.isMouseButtonPressed(Input.MOUSE_BUTTON_LEFT)

        if isPressed then
            -- Drag starten
            if isMouseOverHUD and not SpeedMeterDisplayExt.isDragging then
                SpeedMeterDisplayExt.isDragging  = true
                SpeedMeterDisplayExt.dragOffsetX = mouseX - hudX
                SpeedMeterDisplayExt.dragOffsetY = mouseY - hudY
            end

            -- Drag läuft
            if SpeedMeterDisplayExt.isDragging then
                local newHudX = mouseX - SpeedMeterDisplayExt.dragOffsetX
                local newHudY = mouseY - SpeedMeterDisplayExt.dragOffsetY

                -- *** WICHTIG:  relative Offsets speichern! ***
                local relX = (newHudX - posX) / self:scalePixelToScreenWidth(1)
                local relY = (newHudY - posY) / self:scalePixelToScreenHeight(1)

                SpeedMeterDisplayExt.DEFAULT_POSITION.x = relX
                SpeedMeterDisplayExt.DEFAULT_POSITION.y = relY

                self.customImageOffsetX = self:scalePixelToScreenWidth(relX)
                self.customImageOffsetY = self:scalePixelToScreenHeight(relY)
            end
        else
            -- Drag beendet → Position sichern
            if SpeedMeterDisplayExt.isDragging then
                SpeedMeterDisplayExt.isDragging = false
                SpeedMeterDisplayExt.savePosition()
            end
        end

        -- Prüfen, ob auf eines der Dreiecke geklickt wurde
        if not SpeedMeterDisplayExt.isDragging and isPressed then
            -- Position des linken Dreiecks berechnen
            local triangleLeftX = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_LEFT_OFFSET.x)
            local triangleLeftY = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_LEFT_OFFSET.y)
            
            -- Position des rechten Dreiecks berechnen
            local triangleRightX = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_RIGHT_OFFSET.x)
            local triangleRightY = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_RIGHT_OFFSET.y)
            
            -- Prüfen, ob Maus über dem linken Dreieck ist
            local isOverLeftTriangle = 
                mouseX >= triangleLeftX and 
                mouseX <= triangleLeftX + self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_LEFT_SIZE.width) and
                mouseY >= triangleLeftY and 
                mouseY <= triangleLeftY + self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_LEFT_SIZE.height)
            
            -- Prüfen, ob Maus über dem rechten Dreieck ist
            local isOverRightTriangle = 
                mouseX >= triangleRightX and 
                mouseX <= triangleRightX + self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_RIGHT_SIZE.width) and
                mouseY >= triangleRightY and 
                mouseY <= triangleRightY + self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_RIGHT_SIZE.height)
            
            -- Design wechseln bei Klick auf Dreieck
            if isOverLeftTriangle and not self.leftTriangleWasClicked then
                self.leftTriangleWasClicked = true
                SpeedMeterDisplayExt.switchDesign(false)
            elseif isOverRightTriangle and not self.rightTriangleWasClicked then
                self.rightTriangleWasClicked = true
                SpeedMeterDisplayExt.switchDesign(true)
            end
        else
            -- Reset der Klick-Status wenn Maustaste losgelassen
            self.leftTriangleWasClicked = false
            self.rightTriangleWasClicked = false
        end

        -- Cursor-Icon (Hand vs. Pfeil)
        if g_mouseHandler ~= nil then
            g_mouseHandler:setCursor(isMouseOverHUD and MouseCursor.MOVE or MouseCursor.NORMAL)
        end
    end
end


function SpeedMeterDisplayExt.getInterpolatedAngle(ratio)
    -- Direkter mathematischer Ansatz: Linear von 70° (leer) bis -70° (voll)
    -- Die Gesamtspanne beträgt 140 Grad
    local maxAngle = 70 * math.pi/180  -- Maximalwinkel bei leerem Tank (in Radiant)
    local minAngle = -70 * math.pi/180 -- Minimalwinkel bei vollem Tank (in Radiant)
    
    -- Einfache lineare Interpolation
    local angle = maxAngle - (ratio * (maxAngle - minAngle))
    
    return angle
end

-- SpeedMeterDisplay.draw erweitern
function SpeedMeterDisplay:draw()
    -- Original aufrufen
    SpeedMeterDisplayExt.originalDraw(self)

    -- Unser eigenes Bild zeichnen, aber nur wenn das Fahrzeug AdBlue hat
    if self.vehicle ~= nil and self.isVehicleDrawSafe and self.customImage then
        -- AdBlue-Füllstand auslesen
        local adLvl, adCap = SpeedMeterDisplayExt.getVehicleAdBlueLevelAndCapacity(self.vehicle)
        local hasAdBlue = adLvl ~= nil and adCap ~= nil and adCap > 0
        
        -- Nur anzeigen, wenn das Fahrzeug AdBlue hat (auch im Positionierungsmodus nicht anzeigen, wenn kein AdBlue)
        if hasAdBlue then
            local posX, posY = self:getPosition()
            
            -- HINZUGEFÜGT: Hintergrundbild positionieren
            if self.backgroundImage then
                local bgX = posX + self.customImageOffsetX + 
                           self:scalePixelToScreenWidth(SpeedMeterDisplayExt.BACKGROUND_IMAGE_OFFSET.x)
                local bgY = posY + self.customImageOffsetY + 
                           self:scalePixelToScreenHeight(SpeedMeterDisplayExt.BACKGROUND_IMAGE_OFFSET.y)
                
                self.backgroundImage:setPosition(bgX, bgY)
            end
            
            -- Erstes Bild positionieren
            self.customImage:setPosition(
                posX + self.customImageOffsetX, 
                posY + self.customImageOffsetY)
            
            -- Zweites Bild positionieren und Rotation berechnen
            if self.secondImage then
                local secondImageX = posX + self.customImageOffsetX + 
                                    self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SECOND_IMAGE_OFFSET.x)
                local secondImageY = posY + self.customImageOffsetY + 
                                    self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SECOND_IMAGE_OFFSET.y)
                
                self.secondImage:setPosition(secondImageX, secondImageY)
                
                -- Füllstandsverhältnis berechnen (0.0 bis 1.0)
                local ratio = adLvl / adCap
                
                -- Kritischer Wert - betrachte als 0
                if ratio < 0.01 then 
                    ratio = 0
                end
                
                -- Interpoliere den Winkel aus der Wertetabelle
                local angle = SpeedMeterDisplayExt.getInterpolatedAngle(ratio)
                
                -- Drehpunkt ist die Mitte des Bildes
                local centerX = self.secondImage.width * 0.5
                local centerY = self.secondImage.height * 0.5
                
                -- Rotation anwenden
                self.secondImage:setRotation(angle, centerX, centerY)
--[[
                -- Debug-Status-Anzeige
                if DEBUG then
                    dprint("AdBlue: %0.2f/%0.2f L (%0.1f%%), Winkel: %0.2f Grad", 
                        adLvl, adCap, ratio * 100, angle * 180 / math.pi)
                    
                    -- AdBlue-Status als Text im Spiel (nur im Debug-Modus)
                    if SpeedMeterDisplayExt.isPositioningMode then
                        setTextAlignment(RenderText.ALIGN_LEFT)
                        renderText(0.05 * g_screenWidth, 0.2 * g_screenHeight, 0.016, 
                            string.format("AdBlue: %.1f/%.1f L (%.1f%%)", adLvl, adCap, ratio * 100))
                    end
                end
]]--
            end

            -- Im Positionierungsmodus halbtransparent anzeigen
            if SpeedMeterDisplayExt.isPositioningMode then
                if self.backgroundImage then
                    self.backgroundImage:setColor(1.0, 1.0, 1.0, 0.7)
                end
                self.customImage:setColor(1.0, 1.0, 1.0, 0.7)
                if self.secondImage then
                    self.secondImage:setColor(1.0, 1.0, 1.0, 0.7)
                end
                
                -- Hilfetext anzeigen
                setTextAlignment(RenderText.ALIGN_CENTER)
                renderText(0.5 * g_screenWidth, 0.9 * g_screenHeight, 0.02, 
                    "HUD-Positionierungsmodus aktiv\nPfeiltasten: Bewegen | S: Speichern | Y: Modus beenden")
                setTextAlignment(RenderText.ALIGN_LEFT)
            else
                if self.backgroundImage then
                    self.backgroundImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
                self.customImage:setColor(1.0, 1.0, 1.0, 1.0)
                if self.secondImage then
                    self.secondImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
            end
            
            if self.progressbarMainImage and SpeedMeterDisplayExt.SHOW_PROGRESSBAR then
                local progressX = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.PROGRESSBAR_OFFSET.x)
                local progressY = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.PROGRESSBAR_OFFSET.y)
                self.progressbarMainImage:setPosition(progressX, progressY)
                
                if SpeedMeterDisplayExt.isPositioningMode then
                    self.progressbarMainImage:setColor(0.0, 0.0, 0.0, 0.7)  -- Schwarz mit 70% Deckkraft
                else
                    self.progressbarMainImage:setColor(0.0, 0.0, 0.0, 1.0)  -- Schwarz mit voller Deckkraft
                end
                
            end

            -- Zweite Progressbar positionieren
            if self.progressbarSecondImage and SpeedMeterDisplayExt.SHOW_PROGRESSBAR2 then
                local progress2X = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.PROGRESSBAR2_OFFSET.x)
                local progress2Y = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.PROGRESSBAR2_OFFSET.y)
                
                -- Stark reduzierter kritischer Wert für eine genauere Anzeige
                local crit = 0.01  -- Praktisch nur 0 Liter als kritischer Wert
                
                -- Verhältnis berechnen, selbst für sehr kleine Werte
                local ratio = 0
                if adCap > 0 and adLvl > crit then
                    ratio = adLvl / adCap
                    
                    -- Sehr kleine Werte werden etwas verstärkt dargestellt, damit sie sichtbar sind
                    if ratio < 0.01 then
                        -- Mindesthöhe garantieren - macht aus sehr kleinen Werten mindestens 1% der Anzeige
                        ratio = math.max(ratio, 0.01)
                    end
                end
                
                -- Maximale Höhe der Progressbar
                local maxHeight = self:scalePixelToScreenHeight(SpeedMeterDisplayExt.PROGRESSBAR2_SIZE.height)
                
                -- Skaliere die Höhe entsprechend dem Füllstand
                local barHeight = maxHeight * ratio
                
                -- Setze die neue Höhe der Progressbar
                self.progressbarSecondImage:setDimension(
                    self:scalePixelToScreenWidth(SpeedMeterDisplayExt.PROGRESSBAR2_SIZE.width),
                    barHeight
                )
                
                -- Positioniere den Balken
                self.progressbarSecondImage:setPosition(progress2X, progress2Y)
                
                -- Nur rendern wenn tatsächlich AdBlue vorhanden ist
                if ratio > 0 then
                    local alpha = 1.0
                    
                    -- Bei niedrigem Füllstand blinken lassen
                    if ratio <= 0.25 then
                        alpha = math.abs(math.cos(g_time / 300))  -- Blinken wie im AdblueSpeedMeterPatcher
                        self.progressbarSecondImage:setColor(1.0, 0.0, 0.0, SpeedMeterDisplayExt.isPositioningMode and alpha * 0.7 or alpha)  -- Rot bei niedrigem Stand
                    else
                        self.progressbarSecondImage:setColor(0.0, 0.0, 1.0, SpeedMeterDisplayExt.isPositioningMode and 0.7 or 1.0)  -- Blau bei normalem Stand
                    end
                    
                    -- Im Positionierungsmodus ist die Transparenz bereits in der Farbe integriert
                    self.progressbarSecondImage:render()
                end
            end

            -- AdBlue Fuel-Icon positionieren und rendern
            if self.fuelIconImage and SpeedMeterDisplayExt.SHOW_FUEL_ICON then
                local iconX = posX + self.customImageOffsetX + 
                            self:scalePixelToScreenWidth(SpeedMeterDisplayExt.FUEL_ICON_OFFSET.x)
                local iconY = posY + self.customImageOffsetY + 
                            self:scalePixelToScreenHeight(SpeedMeterDisplayExt.FUEL_ICON_OFFSET.y)
                self.fuelIconImage:setPosition(iconX, iconY)
                
                -- Transparenz im Positionierungsmodus anpassen
                if SpeedMeterDisplayExt.isPositioningMode then
                    self.fuelIconImage:setColor(1.0, 1.0, 1.0, 0.7)
                else
                    self.fuelIconImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
            end

            -- Linkes Dreieck positionieren und anzeigen
            if self.triangleLeftImage and SpeedMeterDisplayExt.SHOW_TRIANGLE_LEFT then
                local triangleLeftX = posX + self.customImageOffsetX + 
                                    self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_LEFT_OFFSET.x)
                local triangleLeftY = posY + self.customImageOffsetY + 
                                    self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_LEFT_OFFSET.y)
                self.triangleLeftImage:setPosition(triangleLeftX, triangleLeftY)
                
                -- Transparenz im Positionierungsmodus anpassen
                if SpeedMeterDisplayExt.isPositioningMode then
                    --self.triangleLeftImage:setColor(1.0, 1.0, 1.0, 0.7)
                else
                    self.triangleLeftImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
                
                self.triangleLeftImage:render()
            end

            -- Rechtes Dreieck positionieren und anzeigen
            if self.triangleRightImage and SpeedMeterDisplayExt.SHOW_TRIANGLE_RIGHT then
                local triangleRightX = posX + self.customImageOffsetX + 
                                    self:scalePixelToScreenWidth(SpeedMeterDisplayExt.TRIANGLE_RIGHT_OFFSET.x)
                local triangleRightY = posY + self.customImageOffsetY + 
                                    self:scalePixelToScreenHeight(SpeedMeterDisplayExt.TRIANGLE_RIGHT_OFFSET.y)
                self.triangleRightImage:setPosition(triangleRightX, triangleRightY)
                
                -- Transparenz im Positionierungsmodus anpassen
                if SpeedMeterDisplayExt.isPositioningMode then
                    --self.triangleRightImage:setColor(1.0, 1.0, 1.0, 0.7)
                else
                    self.triangleRightImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
                
                self.triangleRightImage:render()
            end

            if self.squareImages and #self.squareImages > 0 and SpeedMeterDisplayExt.SHOW_SQUARES then
                -- Positionierung und Darstellung aller Quadrate einheitlich
                for i, square in ipairs(self.squareImages) do
                    local squareX = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SQUARES_START_OFFSET.x)
                    local squareY = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARES_START_OFFSET.y) + 
                                (i-1) * (self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARE_SIZE.height) + 
                                        self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARES_GAP))
                    
                    square:setPosition(squareX, squareY)
                    square:setColor(1.0, 1.0, 1.0, SpeedMeterDisplayExt.isPositioningMode and 0.7 or 1.0)

                    square:render()
                end
            end

            -- Zweite Squares-Reihe positionieren und rendern
            if self.square2Images and #self.square2Images > 0 and SpeedMeterDisplayExt.SHOW_SQUARES2 then
                local fullSquareCount = 0
                
                -- Berechnen, wie viele Quadrate gefüllt werden sollen basierend auf dem AdBlue-Füllstand
                if adCap > 0 then
                    local ratio = adLvl / adCap
                    fullSquareCount = math.min(12, math.ceil(ratio * 12))
                end
                
                -- Umgekehrte Reihenfolge der Quadrate für Anzeige von unten nach oben
                for i = SpeedMeterDisplayExt.SQUARES2_COUNT, 1, -1 do
                    local square = self.square2Images[i]
                    
                    -- Index umrechnen für die Position (SQUARES2_COUNT-i+1 ist der umgekehrte Index von i)
                    local displayIndex = SpeedMeterDisplayExt.SQUARES2_COUNT - i + 1
                    
                    local squareX = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.SQUARES2_START_OFFSET.x)
                    local squareY = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARES2_START_OFFSET.y) + 
                                displayIndex * (self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARE2_SIZE.height) + 
                                        self:scalePixelToScreenHeight(SpeedMeterDisplayExt.SQUARES2_GAP))
                    
                    square:setPosition(squareX, squareY)
                    
                    -- Farbe basierend auf Füllstand setzen
                    if i > SpeedMeterDisplayExt.SQUARES2_COUNT - fullSquareCount then
                        -- Dies ist ein gefülltes Quadrat
                        
                        -- Nur das unterste Quadrat (i=10) rot und blinkend, wenn AdBlue unter 10%
                        if i == SpeedMeterDisplayExt.SQUARES2_COUNT and (adLvl / adCap) <= 0.1 then
                            -- Unterste Quadrat rot und blinkend bei niedrigem Füllstand
                            local alpha = math.abs(math.cos(g_time / 300))
                            square:setColor(1.0, 0.0, 0.0, SpeedMeterDisplayExt.isPositioningMode and alpha * 0.7 or alpha)
                        else
                            -- Alle anderen gefüllten: Blau
                            square:setColor(0.0, 0.0, 1.0, SpeedMeterDisplayExt.isPositioningMode and 0.7 or 1.0)
                        end
                    else
                        -- Leere Quadrate: schwarz
                        square:setColor(0.0, 0.0, 0.0, SpeedMeterDisplayExt.isPositioningMode and 0.5 or 0.7)
                    end
                    
                    square:render()
                end
            end

            -- AdBlue Fuel-Icon 2 positionieren und rendern
            if self.fuelIcon2Image and SpeedMeterDisplayExt.SHOW_FUEL_ICON2 then
                local icon2X = posX + self.customImageOffsetX + 
                            self:scalePixelToScreenWidth(SpeedMeterDisplayExt.FUEL_ICON2_OFFSET.x)
                local icon2Y = posY + self.customImageOffsetY + 
                            self:scalePixelToScreenHeight(SpeedMeterDisplayExt.FUEL_ICON2_OFFSET.y)
                self.fuelIcon2Image:setPosition(icon2X, icon2Y)
                
                if (adLvl / adCap) <= 0.1 then
                    local alpha = math.abs(math.cos(g_time / 300))
                    --self.fuelIcon2Image:setColor(0.7, 0.0, 0.0, 1.0)
                    self.fuelIcon2Image:setColor(0.7, 0.0, 0.0, alpha * 1.0)  -- Rot bei niedrigem Füllstand
                    
                    
                else
                    self.fuelIcon2Image:setColor(0.0, 0.0, 0.0, 0.0)  -- Unsichtbar, wenn AdBlue unter 10%
                end
                self.fuelIcon2Image:render()
            end

            -- AdBlue Backgrund positionieren und rendern
            if self.background2Image and SpeedMeterDisplayExt.SHOW_BACKGROUND2 then
                local iconX = posX + self.customImageOffsetX + 
                            self:scalePixelToScreenWidth(SpeedMeterDisplayExt.BACKGROUND2_IMAGE_OFFSET.x)
                local iconY = posY + self.customImageOffsetY + 
                            self:scalePixelToScreenHeight(SpeedMeterDisplayExt.BACKGROUND2_IMAGE_OFFSET.y)
                self.background2Image:setPosition(iconX, iconY)
                
                -- Transparenz im Positionierungsmodus anpassen
                if SpeedMeterDisplayExt.isPositioningMode then
                    self.fuelIconImage:setColor(1.0, 1.0, 1.0, 0.7)
                else
                    self.fuelIconImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
                self.background2Image:render()
            end

            -- Nur Prozent-Text anzeigen
            if SpeedMeterDisplayExt.SHOW_PERCENT_TEXT then
                local textX = posX + self.customImageOffsetX + 
                            self:scalePixelToScreenWidth(SpeedMeterDisplayExt.PERCENT_TEXT_OFFSET.x)
                local textY = posY + self.customImageOffsetY + 
                            self:scalePixelToScreenHeight(SpeedMeterDisplayExt.PERCENT_TEXT_OFFSET.y)
                
                -- Verhältnis berechnen
                local ratio = 0
                if adCap > 0 then
                    ratio = adLvl / adCap
                    if ratio < 0.01 then ratio = 0 end
                end
                
                -- Prozenttext erstellen
                --local percentText = string.format("%.3f%%", ratio * 100)
                --local percentText = string.format("%d%%", math.floor(ratio * 100))
                local percentText = string.format("%d%%", math.ceil(ratio * 100))
                
                -- Textausrichtung setzen
                setTextAlignment(RenderText.ALIGN_CENTER)
                setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_MIDDLE)
                
                -- Textfarbe basierend auf Füllstand
                local textSize = SpeedMeterDisplayExt.PERCENT_TEXT_SIZE
                if ratio <= 0.25 then
                    -- Kritischer Füllstand: Rot und blinkend wenn sehr niedrig
                    local alpha = 1.0
                    if ratio <= 0.1 then
                        alpha = math.abs(math.cos(g_time / 300))
                    end
                    setTextColor(1.0, 0.0, 0.0, SpeedMeterDisplayExt.isPositioningMode and alpha * 0.7 or alpha)
                else
                    -- Normaler Füllstand: Weiß
                    setTextColor(1.0, 1.0, 1.0, SpeedMeterDisplayExt.isPositioningMode and 0.7 or 1.0)
                end
                
                -- Text rendern
                renderText(textX, textY, textSize, percentText)
                
                -- Texteinstellungen zurücksetzen
                setTextAlignment(RenderText.ALIGN_LEFT)
                setTextVerticalAlignment(RenderText.VERTICAL_ALIGN_TOP)
                setTextColor(1.0, 1.0, 1.0, 1.0)
            end

            -- Zusätzliches Adblue Bild positionieren und anzeigen
            if self.additionalImage and SpeedMeterDisplayExt.SHOW_ADDITIONAL_IMAGE then
                local additionalX = posX + self.customImageOffsetX + 
                                self:scalePixelToScreenWidth(SpeedMeterDisplayExt.ADDITIONAL_IMAGE_OFFSET.x)
                local additionalY = posY + self.customImageOffsetY + 
                                self:scalePixelToScreenHeight(SpeedMeterDisplayExt.ADDITIONAL_IMAGE_OFFSET.y)
                self.additionalImage:setPosition(additionalX, additionalY)
                
                -- Transparenz im Positionierungsmodus anpassen
                if SpeedMeterDisplayExt.isPositioningMode then
                    self.additionalImage:setColor(1.0, 1.0, 1.0, 0.7)
                else
                    self.additionalImage:setColor(1.0, 1.0, 1.0, 1.0)
                end
                
                self.additionalImage:render()
            end

            if self.backgroundImage and SpeedMeterDisplayExt.SHOW_BACKGROUND then
                self.backgroundImage:render()
            end

            if self.customImage and SpeedMeterDisplayExt.SHOW_GAUGE then
                self.customImage:render()
            end

            if self.secondImage and SpeedMeterDisplayExt.SHOW_NEEDLE then
                self.secondImage:render()
            end

            if self.progressbarMainImage and SpeedMeterDisplayExt.SHOW_PROGRESSBAR then
                self.progressbarMainImage:render()
            end

            if self.progressbarSecondImage and SpeedMeterDisplayExt.SHOW_PROGRESSBAR2 then
                self.progressbarSecondImage:render()
            end

            if self.fuelIconImage and SpeedMeterDisplayExt.SHOW_FUEL_ICON then
                self.fuelIconImage:render()
            end
        end
    end
end

-- Position und Design-Einstellung speichern
function SpeedMeterDisplayExt.savePosition()
    -- Stelle sicher, dass der modSettings-Ordner existiert
    createFolder(getUserProfileAppPath() .. "modSettings/")
    
    local xmlFile = createXMLFile("HUDPositionXML", SpeedMeterDisplayExt.CONFIG_FILENAME, "hudSettings")
    
    if xmlFile then
        -- Position speichern
        setXMLFloat(xmlFile, "hudSettings.position.x", SpeedMeterDisplayExt.DEFAULT_POSITION.x)
        setXMLFloat(xmlFile, "hudSettings.position.y", SpeedMeterDisplayExt.DEFAULT_POSITION.y)
        
        -- Design-Einstellung speichern
        setXMLInt(xmlFile, "hudSettings.designIndex", SpeedMeterDisplayExt.currentDesignIndex)
        
        saveXMLFile(xmlFile)
        dprint("Position und Design gespeichert: x=%f, y=%f, Design=%d", 
               SpeedMeterDisplayExt.DEFAULT_POSITION.x,
               SpeedMeterDisplayExt.DEFAULT_POSITION.y,
               SpeedMeterDisplayExt.currentDesignIndex)
               
        delete(xmlFile)
    end
end

-- Gespeicherte Position und Design laden
function SpeedMeterDisplayExt.loadSavedPosition()
    if fileExists(SpeedMeterDisplayExt.CONFIG_FILENAME) then
        local xmlFile = loadXMLFile("HUDPositionXML", SpeedMeterDisplayExt.CONFIG_FILENAME)
        
        if xmlFile then
            -- Position laden
            SpeedMeterDisplayExt.DEFAULT_POSITION.x = getXMLFloat(xmlFile, "hudSettings.position.x") or SpeedMeterDisplayExt.DEFAULT_POSITION.x
            SpeedMeterDisplayExt.DEFAULT_POSITION.y = getXMLFloat(xmlFile, "hudSettings.position.y") or SpeedMeterDisplayExt.DEFAULT_POSITION.y
            
            -- Design-Einstellung laden
            local designIndex = getXMLInt(xmlFile, "hudSettings.designIndex") or 1
            SpeedMeterDisplayExt.currentDesignIndex = designIndex
            
            -- Einstellungen des Designs anwenden
            local design = SpeedMeterDisplayExt.DESIGNS[SpeedMeterDisplayExt.currentDesignIndex]
            if design then
                for key, value in pairs(design.settings) do
                    SpeedMeterDisplayExt[key] = value
                end
                dprint("Design-Einstellung geladen: %s", design.name)
            end
            
            dprint("Position und Design geladen: x=%f, y=%f, Design=%d", 
                   SpeedMeterDisplayExt.DEFAULT_POSITION.x,
                   SpeedMeterDisplayExt.DEFAULT_POSITION.y,
                   SpeedMeterDisplayExt.currentDesignIndex)
            
            delete(xmlFile)
        end
    end
end

function SpeedMeterDisplayExt.switchDesign(next)
    local designCount = #SpeedMeterDisplayExt.DESIGNS
    
    -- Nächstes/vorheriges Design auswählen
    if next then
        SpeedMeterDisplayExt.currentDesignIndex = SpeedMeterDisplayExt.currentDesignIndex % designCount + 1
    else
        SpeedMeterDisplayExt.currentDesignIndex = SpeedMeterDisplayExt.currentDesignIndex - 1
        if SpeedMeterDisplayExt.currentDesignIndex < 1 then 
            SpeedMeterDisplayExt.currentDesignIndex = designCount 
        end
    end
    
    -- Einstellungen des neuen Designs anwenden
    local design = SpeedMeterDisplayExt.DESIGNS[SpeedMeterDisplayExt.currentDesignIndex]
    
    for key, value in pairs(design.settings) do
        SpeedMeterDisplayExt[key] = value
    end
    
    print("AdBlue-Anzeige: Design '" .. design.name .. "' aktiviert")
    
    -- Position und Design speichern
    SpeedMeterDisplayExt.savePosition()
end

print("=== SpeedMeterDisplay-Erweiterung mit mehreren Bildern initialisiert ===")
print("=== Drücke Y für Positionierungsmodus, dann Pfeiltasten zum Bewegen ===")

-- Kamera einfrieren, solange der HUD-Positionierungsmodus läuft
local origLookUpDown    = VehicleCamera.actionEventLookUpDown
local origLookLeftRight = VehicleCamera.actionEventLookLeftRight
local origZoomSmoothly  = VehicleCamera.zoomSmoothly

function VehicleCamera:actionEventLookUpDown(actionName, inputValue, ...)
    if SpeedMeterDisplayExt and SpeedMeterDisplayExt.isPositioningMode then
        return                               -- Eingabe ignorieren
    end
    origLookUpDown(self, actionName, inputValue, ...)
end

function VehicleCamera:actionEventLookLeftRight(actionName, inputValue, ...)
    if SpeedMeterDisplayExt and SpeedMeterDisplayExt.isPositioningMode then
        return
    end
    origLookLeftRight(self, actionName, inputValue, ...)
end

function VehicleCamera:zoomSmoothly(delta, ...)
    if SpeedMeterDisplayExt and SpeedMeterDisplayExt.isPositioningMode then
        return
    end
    origZoomSmoothly(self, delta, ...)
end
