-- FS25 Bale Finder 
-- Shows bales as colored hotspots on map & minimap.
-- Colors: Straw=yellow, Hay=light green, Grass=dark green, Silage=brown, Other=gray

-- =========================
-- Hotspot subclass
-- =========================
BaleFinderHotspot = {}
local BaleFinderHotspot_mt = Class(BaleFinderHotspot, MapHotspot)

function BaleFinderHotspot.new(rootNode, r, g, b)
  local self = MapHotspot.new(BaleFinderHotspot_mt)

  -- simple square icon; color carries meaning
  self.width, self.height = getNormalizedScreenValues(40, 40)
  if g_overlayManager and g_overlayManager.createOverlay then
    self.icon = g_overlayManager:createOverlay("mapHotspots.other", 0, 0, self.width, self.height)
  end

  if self.setColor and r and g and b then
    self:setColor(r, g, b)
  end

  if rootNode and self.setWorldPosition then
    local x, y, z = getWorldTranslation(rootNode)
    self:setWorldPosition(x, z)
  end

  if self.setVisible then
    self:setVisible(true)
  end

  return self
end

function BaleFinderHotspot:getCategory()
  return MapHotspot.CATEGORY_OTHER
end

-- =========================
-- Controller
-- =========================
local COL = {
  STRAW          = {1.00, 0.90, 0.20},  -- yellow
  HAY            = {0.70, 0.90, 0.50},  -- light green
  GRASS          = {0.20, 0.50, 0.20},  -- dark green
  SILAGE         = {0.45, 0.30, 0.15},  -- brown
  CHAFF          = {1.00, 0.60, 0.20},  -- amber
  SUGARBEET_CUT  = {0.80, 0.20, 0.60},  -- magenta
  FORAGE         = {0.10, 0.65, 0.70},  -- turquoise
  COTTON         = {0.95, 0.95, 1.00},  -- white

  OTHER          = {0.80, 0.80, 0.80}   -- gray
}

local TICK_MS        = 300
local POS_REFRESH_MS = 1000
local MAX_HOTSPOTS   = 800

BaleFinder = {
  initDone  = false,
  clockMs   = 0,
  tickAcc   = 0,
  ft        = {},
  hsByNode  = {},   -- nodeId -> hotspot
  nextPosAt = {},   -- nodeId -> time(ms)
  scanList  = {},
  scanIdx   = 1
}

-- helpers
local function okNode(id)
  if entityExists ~= nil then
    return (id ~= nil and id ~= 0 and entityExists(id))
  end
  return (id ~= nil and id ~= 0)
end

local function isBale(o)
  if not o then return false end
  if o.isa then
    local ok, res = pcall(function() return o:isa(Bale) end)
    if ok and res then return true end
  end
  return (o.getFillType ~= nil and o.nodeId ~= nil)
end

local function colorFor(ft, idx)
  if ft ~= nil then
    if idx.STRAW and ft == idx.STRAW then return COL.STRAW end
    if (idx.DRYGRASS and ft == idx.DRYGRASS) or (idx.HAY and ft == idx.HAY) then return COL.HAY end
    if (idx.GRASS and ft == idx.GRASS) or (idx.GRASS_WINDROW and ft == idx.GRASS_WINDROW) then return COL.GRASS end
    if idx.SILAGE and ft == idx.SILAGE then return COL.SILAGE end
    if idx.CHAFF and ft == idx.CHAFF then return COL.CHAFF or {1.00, 0.60, 0.20} end
    if idx.SUGARBEETCUT and ft == idx.SUGARBEETCUT then return COL.SUGARBEET_CUT end
    if idx.FORAGE and ft == idx.FORAGE then return COL.FORAGE or {0.10, 0.65, 0.70} end
    if idx.COTTON and ft == idx.COTTON then return COL.COTTON or {0.95, 0.95, 1.00} end
  end
  return COL.OTHER
end

-- init & lifecycle
function BaleFinder:init()
  self.initDone = true

  local ftm = g_fillTypeManager
  local function idx(name)
    if not ftm then return nil end
    local v = nil
    if ftm.getFillTypeIndexByName then
      local ok, val = pcall(function() return ftm:getFillTypeIndexByName(name) end)
      if ok then v = val end
    end
    if v == nil and ftm.getFillTypeByName then
      local ok, val = pcall(function() return ftm:getFillTypeByName(name) end)
      if ok then v = val end
    end
    return v
  end

  self.ft.STRAW         = idx("STRAW")
  self.ft.DRYGRASS      = idx("DRYGRASS_WINDROW") or idx("HAY")
  self.ft.HAY           = idx("HAY")
  self.ft.GRASS         = idx("GRASS")
  self.ft.GRASS_WINDROW = idx("GRASS_WINDROW")
  self.ft.SILAGE        = idx("SILAGE")
  self.ft.CHAFF         = idx("CHAFF")
  self.ft.SUGARBEETCUT  = idx("SUGARBEET_CUT")
  self.ft.FORAGE        = idx("FORAGE")            
  self.ft.COTTON        = idx("COTTON")
end

function BaleFinder:onMissionLoaded()
  self.scanList, self.scanIdx = {}, 1
  local nto = g_currentMission and g_currentMission.nodeToObject or {}
  for nodeId,_ in pairs(nto) do
    self.scanList[#self.scanList+1] = nodeId
  end
end

function BaleFinder:update(dt)
  if g_currentMission == nil or not g_currentMission:getIsClient() then return end
  if not self.initDone then self:init() end

  self.clockMs = self.clockMs + dt
  self.tickAcc = self.tickAcc + dt
  if self.tickAcc < TICK_MS then return end
  self.tickAcc = 0

  if (self.scanIdx > #self.scanList) or (#self.scanList == 0) then
    self:onMissionLoaded()
  end

  local nto = g_currentMission.nodeToObject or {}
  local step, maxStep = 0, 200

  while self.scanIdx <= #self.scanList and step < maxStep do
    local nodeId = self.scanList[self.scanIdx]; self.scanIdx = self.scanIdx + 1; step = step + 1
    local obj = nto[nodeId]
    if obj and okNode(nodeId) and isBale(obj) then
      self:addOrUpdate(obj)
    end
  end

  local isys = g_currentMission and g_currentMission.itemSystem
  if isys and isys.items then
    for _, item in pairs(isys.items) do
      local o = item.object
      if o and okNode(o.nodeId) and isBale(o) then
        self:addOrUpdate(o)
      end
    end
  end

  self:cleanup()
end

-- hotspot ops
function BaleFinder:addOrUpdate(bale)
  local nodeId = bale.nodeId
  if not okNode(nodeId) then return end

  local okFT, ftIdx = pcall(function() return bale:getFillType() end)
  if not okFT then return end

  local hs = self.hsByNode[nodeId]
  if not hs then
    if self:countHotspots() >= MAX_HOTSPOTS then return end

    local c = colorFor(ftIdx, self.ft)
    hs = BaleFinderHotspot.new(nodeId, c[1], c[2], c[3])
    if not hs then return end

    if bale.getOwnerFarmId and hs.setOwnerFarmId then
      local okFarm, farmId = pcall(function() return bale:getOwnerFarmId() end)
      if okFarm and farmId then hs:setOwnerFarmId(farmId) end
    end

    if g_currentMission and g_currentMission.addMapHotspot then
      pcall(function() g_currentMission:addMapHotspot(hs) end)
    end

    self.hsByNode[nodeId] = hs
    self.nextPosAt[nodeId] = self.clockMs
  else
    local t = self.clockMs
    if (t - (self.nextPosAt[nodeId] or 0)) >= POS_REFRESH_MS then
      if hs.setWorldPosition and okNode(nodeId) then
        local x, y, z = getWorldTranslation(nodeId)
        hs:setWorldPosition(x, z)
      end
      self.nextPosAt[nodeId] = t
    end

    local c = colorFor(ftIdx, self.ft)
    if hs.setColor then
      hs:setColor(c[1], c[2], c[3])
    end
  end
end

function BaleFinder:cleanup()
  local nto = g_currentMission and g_currentMission.nodeToObject or {}
  for nodeId, hs in pairs(self.hsByNode) do
    local o = nto[nodeId]
    if (not okNode(nodeId)) or (not o) or (not isBale(o)) then
      if g_currentMission and g_currentMission.removeMapHotspot then
        pcall(function() g_currentMission:removeMapHotspot(hs) end)
      end
      self.hsByNode[nodeId] = nil
      self.nextPosAt[nodeId] = nil
    end
  end
end

function BaleFinder:countHotspots()
  local n = 0
  for _ in pairs(self.hsByNode) do n = n + 1 end
  return n
end

-- hooks
local function onMissionFinishedLoad(mission, _)
  if mission and mission.getIsClient and mission:getIsClient() then
    if not BaleFinder.initDone then BaleFinder:init() end
    BaleFinder:onMissionLoaded()
  end
end

Mission00.loadMission00Finished = Utils.appendedFunction(Mission00.loadMission00Finished, onMissionFinishedLoad)
addModEventListener({ update = function(_, dt) BaleFinder:update(dt) end })
